#
from micropython import const
import time
#
import bluetooth
from BluetoothBLE import BleAdvertisingPayload
#
BT_IRQ_CENTRAL_CONNECT = const(1)
BT_IRQ_CENTRAL_DISCONNECT = const(2)
BT_IRQ_GATTS_WRITE = const(3)

BT_FLAG_WRITE = const(0x0008)
BT_FLAG_NOTIFY = const(0x0010)
#
_UART_UUID = bluetooth.UUID("6E400001-B5A3-F393-E0A9-E50E24DCCA9E")
_UART_TX = (
    bluetooth.UUID("6E400003-B5A3-F393-E0A9-E50E24DCCA9E"),
    BT_FLAG_NOTIFY,
)
_UART_RX = (
    bluetooth.UUID("6E400002-B5A3-F393-E0A9-E50E24DCCA9E"),
    BT_FLAG_WRITE,
)
_UART_SERVICE = (
    _UART_UUID,
    (_UART_TX, _UART_RX),
)
#
# org.bluetooth.characteristic.gap.appearance.xml
BT_ADV_APPEARANCE_GENERIC_COMPUTER = const(128)
#
class CBtUart:
    def __init__(self, devicename="ESP32BtUart", rxbuf=100):
        self.DeviceName = devicename
        self._ble = bluetooth.BLE()
        self._ble.active(True)
        self._ble.irq(self._irq)
        ((self._tx_handle, self._rx_handle),) = self._ble.gatts_register_services((_UART_SERVICE,))
        # Increase the size of the rx buffer and enable append mode.
        self._ble.gatts_set_buffer(self._rx_handle, rxbuf, True)
        self._connections = set()
        self._rx_buffer = bytearray()
        self._handler = None
        # Optionally add services=[_UART_UUID], but this is likely to make the payload too large.
        self._payload = BleAdvertisingPayload(name=self.DeviceName, 
                                    appearance=BT_ADV_APPEARANCE_GENERIC_COMPUTER)
        self._advertise()

    def SetIrqHandler(self, irqhandler):
        self._handler = irqhandler

    def _irq(self, event, data):
        # Track connections so we can send notifications.
        if event == BT_IRQ_CENTRAL_CONNECT:
            conn_handle, _, _ = data
            self._connections.add(conn_handle)
        elif event == BT_IRQ_CENTRAL_DISCONNECT:
            conn_handle, _, _ = data
            if conn_handle in self._connections:
                self._connections.remove(conn_handle)
            # Start advertising again to allow a new connection.
            self._advertise()
        elif event == BT_IRQ_GATTS_WRITE:
            conn_handle, value_handle = data
            if conn_handle in self._connections and value_handle == self._rx_handle:
                self._rx_buffer += self._ble.gatts_read(self._rx_handle)
                if self._handler:
                    self._handler()

    def any(self):
        return len(self._rx_buffer)

    def read(self, sz=None):
        if not sz:
            sz = len(self._rx_buffer)
        result = self._rx_buffer[0:sz]
        self._rx_buffer = self._rx_buffer[sz:]
        return result

    def write(self, data):
        for conn_handle in self._connections:
            self._ble.gatts_notify(conn_handle, self._tx_handle, data)

    def close(self):
        for conn_handle in self._connections:
            self._ble.gap_disconnect(conn_handle)
        self._connections.clear()
    #
    def _advertise(self, interval_us=500000):
        self._ble.gap_advertise(interval_us, adv_data=self._payload)
    #