#
from micropython import const
import time
#
import bluetooth
from BluetoothBLE import BleAdvertisingPayload
#
BT_IRQ_CENTRAL_CONNECT = const(1)
BT_IRQ_CENTRAL_DISCONNECT = const(2)
BT_IRQ_GATTS_WRITE = const(3)

BT_FLAG_WRITE = const(0x0008)
BT_FLAG_NOTIFY = const(0x0010)
#
_UART_UUID = bluetooth.UUID("6E400001-B5A3-F393-E0A9-E50E24DCCA9E")
_UART_TX = (
    bluetooth.UUID("6E400003-B5A3-F393-E0A9-E50E24DCCA9E"),
    BT_FLAG_NOTIFY,
)
_UART_RX = (
    bluetooth.UUID("6E400002-B5A3-F393-E0A9-E50E24DCCA9E"),
    BT_FLAG_WRITE,
)
_UART_SERVICE = (
    _UART_UUID,
    (_UART_TX, _UART_RX),
)
#
# org.bluetooth.characteristic.gap.appearance.xml
BT_ADV_APPEARANCE_GENERIC_COMPUTER = const(128)
#
class CBtUart:
    def __init__(self, devicename="ESP32BtUart", rxbuf=100):
        self.DeviceName = devicename
        self.BLE = bluetooth.BLE()
        self.BLE.active(True)
        self.BLE.irq(self._irq)
        ((self._tx_handle, self._rx_handle),) = self.BLE.gatts_register_services((_UART_SERVICE,))
        # Increase the size of the rx buffer and enable append mode.
        self.BLE.gatts_set_buffer(self._rx_handle, rxbuf, True)
        self.Connections = set()
        self.RxBuffer = bytearray()
        self.RxLine = ''
        #self.IrqHandler = None
        self.OnLineReceived = None        
        self.OnLineTransmitted = None
        # Optionally add services=[_UART_UUID], but this is likely to make the payload too large.
        self._payload = BleAdvertisingPayload(name=self.DeviceName, 
                                              appearance=BT_ADV_APPEARANCE_GENERIC_COMPUTER)
        self._advertise()
    #
    def SetOnLineReceived(self, onlinereceived):
        self.OnLineReceived = onlinereceived
    def SetOnLineTransmitted(self, onlinetransmitted):
        self.OnLineTransmitted = onlinetransmitted

    # def SetIrqHandler(self, irqhandler):
    #     self.IrqHandler = irqhandler

    def _irq(self, event, data):
        # Track connections so we can send notifications.
        if event == BT_IRQ_CENTRAL_CONNECT:
            conn_handle, _, _ = data
            self.Connections.add(conn_handle)
        elif event == BT_IRQ_CENTRAL_DISCONNECT:
            conn_handle, _, _ = data
            if conn_handle in self.Connections:
                self.Connections.remove(conn_handle)
            # Start advertising again to allow a new connection.
            self._advertise()
        elif event == BT_IRQ_GATTS_WRITE:
            conn_handle, value_handle = data
            if conn_handle in self.Connections and value_handle == self._rx_handle:
                self.RxBuffer += self.BLE.gatts_read(self._rx_handle)
                while (0 < len(self.RxBuffer)):
                    C = self.read(1)
                    if ((b'\n' == C) or (b'\r' == C)):
                        if (0 < len(self.RxLine)):
                            if (None != self.OnLineReceived):
                                self.OnLineReceived(self.RxLine)
                            self.RxLine = ''
                    else:
                        self.RxLine += C.decode()

    def any(self):
        return len(self.RxBuffer)

    def read(self, sz=None):
        if not sz:
            sz = len(self.RxBuffer)
        result = self.RxBuffer[0:sz]
        self.RxBuffer = self.RxBuffer[sz:]
        return result

    def write(self, data):
        for conn_handle in self.Connections:
            self.BLE.gatts_notify(conn_handle, self._tx_handle, data)

    def close(self):
        for conn_handle in self.Connections:
            self.BLE.gap_disconnect(conn_handle)
        self.Connections.clear()
    #
    def _advertise(self, interval_us=500000):
        self.BLE.gap_advertise(interval_us, adv_data=self._payload)
    #