
from micropython import const
import bluetooth
#
import Thread as THR
import Lines as LIN
import BtBase
#
CR = const(13)
LF = const(10)
#
_IRQ_CENTRAL_CONNECT = const(1)
_IRQ_CENTRAL_DISCONNECT = const(2)
_IRQ_GATTS_WRITE = const(3)
#
_FLAG_WRITE = const(0x0008)
_FLAG_NOTIFY = const(0x0010)
#
_UART_UUID = bluetooth.UUID("6E400001-B5A3-F393-E0A9-E50E24DCCA9E")
_UART_TX = (
    bluetooth.UUID("6E400003-B5A3-F393-E0A9-E50E24DCCA9E"),
    _FLAG_NOTIFY,
)
_UART_RX = (
    bluetooth.UUID("6E400002-B5A3-F393-E0A9-E50E24DCCA9E"),
    _FLAG_WRITE,
)
_UART_SERVICE = (
    _UART_UUID,
    (_UART_TX, _UART_RX),
)
#
# org.bluetooth.characteristic.gap.appearance.xml
_ADV_APPEARANCE_GENERIC_COMPUTER = const(128)
#
class CBtUart():
    def __init__(self, ble, ontxline, onrxline, name="Esp32BtUart", rxbuffersize=100):
        self._ble = ble
        self._ble.active(True)
        self._ble.irq(self._irq)
        ((self._tx_handle, self._rx_handle),) = self._ble.gatts_register_services((_UART_SERVICE,))
        # Increase the size of the rx buffer and enable append mode.
        self._ble.gatts_set_buffer(self._rx_handle, rxbuffersize, True)
        self._connections = set()
        self.RxBuffer = bytearray()
        self.SizeLine = 0
        self.OnTxLine = ontxline
        self.OnRxLine = onrxline
        self.RxLines = LIN.CLines()
        self.TxLines = LIN.CLines()
        # Optionally add services=[_UART_UUID], but this is likely to make the payload too large.
        self._payload = BtBase.AdvertisingPayload(name=name, appearance=_ADV_APPEARANCE_GENERIC_COMPUTER)
        self._advertise()
        self.Thread = THR.CThread(self.CBOnStart, self.CBOnBusy, self.CBOnAbort, self.CBOnEnd)
        return
    #
    def _irq(self, event, data):
        # Track connections so we can send notifications.
        if event == _IRQ_CENTRAL_CONNECT:
            conn_handle, _, _ = data
            self._connections.add(conn_handle)
        elif event == _IRQ_CENTRAL_DISCONNECT:
            conn_handle, _, _ = data
            if conn_handle in self._connections:
                self._connections.remove(conn_handle)
            # Start advertising again to allow a new connection.
            self._advertise()
        elif event == _IRQ_GATTS_WRITE:
            conn_handle, value_handle = data
            if ((conn_handle in self._connections) and (value_handle == self._rx_handle)):
                RxData = self._ble.gatts_read(self._rx_handle)
                # debug print('RxData[{}]'.format(RxData))
                for BValue in RxData:
                    if (CR == BValue) or (LF == BValue):
                        # debug print('RxBuffer[{}]'.format(self.RxBuffer))
                        RxLine = (self.RxBuffer[0:self.SizeLine]).decode('utf-8')
                        # debug print('RxLine[{}]'.format(RxLine))
                        if (0 < self.SizeLine):
                            # debug print('RxLines.append[{}]'.format(RxLine))
                            self.RxLines.Push(RxLine)
                            # debug print(self.RxLines)
                            self.RxBuffer = self.RxBuffer[self.SizeLine:-1]
                            self.SizeLine = 0
                    else:
                        self.SizeLine += 1
                        self.RxBuffer.append(BValue)
    #
    def _advertise(self, interval_us=500000):
        self._ble.gap_advertise(interval_us, adv_data=self._payload)
    #---------------------------------------------------------
    #   Callback Thread
    #---------------------------------------------------------
    def CBOnStart(self, thread):
        return
    def CBOnBusy(self, thread):
        while (THR.stBusy == thread.State):
            # zero: RxData - rxdata in _irq-methode
            # first: RxData - callback
            if (0 < self.RxLines.Count()):
               if (None != self.OnRxLine):
                    RxLine = self.RxLines.Pop()
                    self.OnRxLine(RxLine)
            # second: TxData - send / callback
            if (0 < self.TxLines.Count()):
                TxLine = ''
                TxLine = self.TxLines.Pop()                                    
                if (None != self.OnTxLine):                    
                    TxLine = self.OnTxLine(TxLine)
                TxLine += '\r\n'
                for conn_handle in self._connections:
                    self._ble.gatts_notify(conn_handle, self._tx_handle, TxLine)
        return
    def CBOnAbort(self, thread):
        return
    def CBOnEnd(self, thread):
        return
    #---------------------------------------------------------
    #   Handler
    #---------------------------------------------------------
    def Open(self):
        self.Thread.Start()
        return 
    #
    def Close(self):
        self.Thread.Abort()
        for conn_handle in self._connections:
            self._ble.gap_disconnect(conn_handle)
        self._connections.clear()
        return
    #
    def TxLine(self, txline):
        self.TxLines.Push(txline)
        return txline
    #
    def RxLine(self):
        return self.RxLines.Pop()
    #
#
