"""Provides an API for talking to HD44780 compatible character LCDs."""
#
import time
#
class CLCDisplay:
    # HD44780 LCD controller command set
    LCD_CLR = 0x01              # DB0: clear display
    LCD_HOME = 0x02             # DB1: return to home position
    #
    LCD_ENTRY_MODE = 0x04       # DB2: set entry mode
    LCD_ENTRY_INC = 0x02        # --DB1: increment
    LCD_ENTRY_SHIFT = 0x01      # --DB0: shift
    #
    LCD_ON_CTRL = 0x08          # DB3: turn lcd/cursor on
    LCD_ON_DISPLAY = 0x04       # --DB2: turn display on
    LCD_ON_CURSOR = 0x02        # --DB1: turn cursor on
    LCD_ON_BLINK = 0x01         # --DB0: blinking cursor
    #
    LCD_MOVE = 0x10             # DB4: move cursor/display
    LCD_MOVE_DISP = 0x08        # --DB3: move display (0-> move cursor)
    LCD_MOVE_RIGHT = 0x04       # --DB2: move right (0-> left)
    #
    LCD_FUNCTION = 0x20         # DB5: function set
    LCD_FUNCTION_8BIT = 0x10    # --DB4: set 8BIT mode (0->4BIT mode)
    LCD_FUNCTION_2LINES = 0x08  # --DB3: two lines (0->one line)
    LCD_FUNCTION_10DOTS = 0x04  # --DB2: 5x10 font (0->5x7 font)
    LCD_FUNCTION_RESET = 0x30   # See "Initializing by Instruction" section
    #
    LCD_CGRAM = 0x40            # DB6: set CG RAM address
    LCD_DDRAM = 0x80            # DB7: set DD RAM address
    #
    LCD_RS_CMD = 0
    LCD_RS_DATA = 1
    #
    LCD_RW_WRITE = 0
    LCD_RW_READ = 1
    #
    def __init__(self, num_lines, num_columns):
        self.num_lines = num_lines
        if self.num_lines > 4:
            self.num_lines = 4
        self.num_columns = num_columns
        if self.num_columns > 40:
            self.num_columns = 40
        self.CursorY = 0
        self.CursorX = 0
        self.ImpliedNewline = False
        self.Backlight = True
        self.DisplayOff()
        self.BacklightOn()
        self.Clear()
        self.hal_write_command(self.LCD_ENTRY_MODE | self.LCD_ENTRY_INC)
        self.HideCursor()
        self.DisplayOn()
    #
    def Clear(self):
        self.hal_write_command(self.LCD_CLR)
        self.hal_write_command(self.LCD_HOME)
        self.CursorY = 0
        self.CursorX = 0
    #
    def ShowCursor(self):
        self.hal_write_command(self.LCD_ON_CTRL | self.LCD_ON_DISPLAY |
                               self.LCD_ON_CURSOR)
    #
    def HideCursor(self):
        self.hal_write_command(self.LCD_ON_CTRL | self.LCD_ON_DISPLAY)
    #
    def BlinkCursorOn(self):
        self.hal_write_command(self.LCD_ON_CTRL | self.LCD_ON_DISPLAY |
                               self.LCD_ON_CURSOR | self.LCD_ON_BLINK)
    #
    def BlinkCursorOff(self):
        self.hal_write_command(self.LCD_ON_CTRL | self.LCD_ON_DISPLAY |
                               self.LCD_ON_CURSOR)
    #
    def DisplayOn(self):
        self.hal_write_command(self.LCD_ON_CTRL | self.LCD_ON_DISPLAY)
    #
    def DisplayOff(self):
        self.hal_write_command(self.LCD_ON_CTRL)
    #
    def BacklightOn(self):
        self.backlight = True
        self.hal_backlight_on()
    #
    def BacklightOff(self):
        self.backlight = False
        self.hal_backlight_off()
    #
    def MoveTo(self, cursory, cursorx):
        self.CursorY = cursory
        self.CursorX = cursorx
        Address = cursorx & 0x3f
        if cursory & 1:
            Address += 0x40    # Lines 1 & 3 add 0x40
        if cursory & 2:     # Lines 2 & 3 add number of columns
            Address += self.num_columns
        self.hal_write_command(self.LCD_DDRAM | Address)
    #
    def PutCharacter(self, character):
        if character == '\n':
            if self.implied_newline:
                self.implied_newline = False
            else:
                self.CursorX = self.num_columns
        else:
            self.hal_write_data(ord(character))
            self.CursorX += 1
        if self.CursorX >= self.num_columns:
            self.CursorX = 0
            self.CursorY += 1
            self.implied_newline = (character != '\n')
        if self.CursorY >= self.num_lines:
            self.CursorY = 0
        self.MoveTo(self.CursorY, self.CursorX)
    #
    def PutText(self, text):
        for C in text:
            self.PutCharacter(C)
    #
    def custom_char(self, location, charmap):
        location &= 0x7
        self.hal_write_command(self.LCD_CGRAM | (location << 3))
        self.hal_sleep_us(40)
        for i in range(8):
            self.hal_write_data(charmap[i])
            self.hal_sleep_us(40)
        self.MoveTo(self.cursor_y, self.cursor_x)
    #
    def hal_backlight_on(self):
        pass
    #
    def hal_backlight_off(self):
        pass
    #
    def hal_write_command(self, cmd):
        raise NotImplementedError
    #
    def hal_write_data(self, data):
        raise NotImplementedError
    #
    def hal_sleep_us(self, usecs):
        time.sleep_us(usecs)  
    #
#