#
"""Implements a HD44780 character LCD connected via PCF8574 on I2C"""
#
from LCDisplay import CLCDisplay
from time import sleep_ms
#
# The PCF8574 has a jumper selectable address: 0x20 - 0x27
DEFAULT_I2C_ADDR = 0x27
#
MASK_RS = 0x01
MASK_RW = 0x02
MASK_E = 0x04
SHIFT_BACKLIGHT = 3
SHIFT_DATA = 4
#
class CLCDisplayI2C(CLCDisplay):
    #
    def __init__(self, i2c, i2caddress, countrows, countcolumns):
        self.I2C = i2c
        self.I2CAddress = i2caddress
        self.I2C.writeto(self.I2CAddress, bytearray([0]))
        sleep_ms(20)   # Allow LCD time to powerup
        # Send reset 3 times
        self.hal_write_init_nibble(self.LCD_FUNCTION_RESET)
        sleep_ms(5)    # need to delay at least 4.1 msec
        self.hal_write_init_nibble(self.LCD_FUNCTION_RESET)
        sleep_ms(1)
        self.hal_write_init_nibble(self.LCD_FUNCTION_RESET)
        sleep_ms(1)
        # Put LCD into 4 bit mode
        self.hal_write_init_nibble(self.LCD_FUNCTION)
        sleep_ms(1)
        CLCDisplay.__init__(self, countrows, countcolumns)
        cmd = self.LCD_FUNCTION
        if countrows > 1:
            cmd |= self.LCD_FUNCTION_2LINES
        self.hal_write_command(cmd)
    #
    def hal_write_init_nibble(self, nibble):
        byte = ((nibble >> 4) & 0x0f) << SHIFT_DATA
        self.I2C.writeto(self.I2CAddress, bytearray([byte | MASK_E]))
        self.I2C.writeto(self.I2CAddress, bytearray([byte]))
    #
    def hal_backlight_on(self):
        self.I2C.writeto(self.I2CAddress, bytearray([1 << SHIFT_BACKLIGHT]))
    #
    def hal_backlight_off(self):
        self.I2C.writeto(self.I2CAddress, bytearray([0]))
    #
    def hal_write_command(self, cmd):
        """Data is latched on the falling edge of E."""
        byte = ((self.Backlight << SHIFT_BACKLIGHT) | (((cmd >> 4) & 0x0f) << SHIFT_DATA))
        self.I2C.writeto(self.I2CAddress, bytearray([byte | MASK_E]))
        self.I2C.writeto(self.I2CAddress, bytearray([byte]))
        byte = ((self.Backlight << SHIFT_BACKLIGHT) | ((cmd & 0x0f) << SHIFT_DATA))
        self.I2C.writeto(self.I2CAddress, bytearray([byte | MASK_E]))
        self.I2C.writeto(self.I2CAddress, bytearray([byte]))
        if cmd <= 3: # The home and clear commands require a worst case delay of 4.1 msec
            sleep_ms(5)
    #
    def hal_write_data(self, data):
        byte = (MASK_RS | (self.Backlight << SHIFT_BACKLIGHT) | (((data >> 4) & 0x0f) << SHIFT_DATA))
        self.I2C.writeto(self.I2CAddress, bytearray([byte | MASK_E]))
        self.I2C.writeto(self.I2CAddress, bytearray([byte]))
        byte = (MASK_RS | (self.Backlight << SHIFT_BACKLIGHT) | ((data & 0x0f) << SHIFT_DATA))
        self.I2C.writeto(self.I2CAddress, bytearray([byte | MASK_E]))
        self.I2C.writeto(self.I2CAddress, bytearray([byte]))
    #
#    